from django.shortcuts import render, redirect, get_object_or_404
from django.views import View
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib import messages
from .models import Garment
from services.models import Service

class GarmentManagementView(LoginRequiredMixin, View):
    def get(self, request):
        garments = Garment.objects.all()
        services = Service.objects.filter(status='active')
        return render(request, 'pages/garment-management/garment_management.html', {'garments': garments, 'services': services})

class GarmentCreateView(LoginRequiredMixin, View):
    def get(self, request):
        services = Service.objects.filter(status='active')
        from category.models import Category
        categories = Category.objects.filter(trash=False, status='active')
        default_services = ['Dry cleaning', 'Laundry', 'Steam Iron']
        return render(request, 'pages/garment-management/garment_create.html', {
            'services': services,
            'categories': categories,
            'default_services': default_services
        })

    def post(self, request):
        name = request.POST.get('name')
        status = request.POST.get('status')
        garment_type = request.POST.get('garment_type', 'normal')
        description = request.POST.get('description')
        trash = bool(request.POST.get('trash'))
        category_id = request.POST.get('category')
        from category.models import Category
        category = Category.objects.filter(id=category_id).first() if category_id else None
        service_ids = request.POST.getlist('service')
        prices = request.POST.getlist('price')
        service_prices = []
        for service_id, price in zip(service_ids, prices):
            if service_id and price:
                service_prices.append({"service_id": int(service_id), "price": float(price)})
        garment = Garment.objects.create(
            name=name,
            status=status,
            garment_type=garment_type,
            description=description,
            trash=trash,
            category=category,
            service_prices=service_prices
        )
        messages.success(request, 'Garment created successfully!')
        return redirect('garment-management')

class GarmentDetailViewWeb(LoginRequiredMixin, View):
    def get(self, request, pk):
        garment = get_object_or_404(Garment, pk=pk)
        from services.models import Service
        # Build a list of dicts with service_name and price for display
        service_price_display = []
        for sp in garment.service_prices:
            try:
                service = Service.objects.get(id=sp["service_id"])
                service_price_display.append({"service_name": service.service_name, "price": sp["price"]})
            except Service.DoesNotExist:
                continue
        return render(request, 'pages/garment-management/garment_detail.html', {'garment': garment, 'service_price_display': service_price_display})

class GarmentEditView(LoginRequiredMixin, View):
    def get(self, request, pk):
        garment = get_object_or_404(Garment, pk=pk)
        services = Service.objects.filter(status='active')
        from category.models import Category
        categories = Category.objects.filter(trash=False, status='active')
        return render(request, 'pages/garment-management/garment_edit.html', {'garment': garment, 'services': services, 'categories': categories})

    def post(self, request, pk):
        garment = get_object_or_404(Garment, pk=pk)
        garment.name = request.POST.get('name')
        garment.status = request.POST.get('status')
        garment.garment_type = request.POST.get('garment_type', 'normal')
        garment.description = request.POST.get('description')
        garment.trash = bool(request.POST.get('trash'))
        category_id = request.POST.get('category')
        from category.models import Category
        garment.category = Category.objects.filter(id=category_id).first() if category_id else None
        service_ids = request.POST.getlist('service')
        prices = request.POST.getlist('price')
        service_prices = []
        for service_id, price in zip(service_ids, prices):
            if service_id and price:
                service_prices.append({"service_id": int(service_id), "price": float(price)})
        garment.service_prices = service_prices
        garment.save()
        messages.success(request, 'Garment updated successfully!')
        return redirect('garment-management')

class GarmentDeleteView(LoginRequiredMixin, View):
    def post(self, request, pk):
        garment = get_object_or_404(Garment, pk=pk)
        garment.trash = True
        garment.deleted_at = None
        garment.save()
        messages.success(request, 'Garment deleted (trashed) successfully!')
        return redirect('garment-management')
