from rest_framework import generics, permissions, status
from django.db import models
from common.response_helper import ResponseHelper
from .models import Booking
from .serializers import BookingSerializer


class BookingListCreateView(generics.ListCreateAPIView):
    serializer_class = BookingSerializer
    permission_classes = [permissions.IsAuthenticated]

    def get_queryset(self):
        user = self.request.user
        qs = Booking.objects.all()
        if user.is_staff or user.is_superuser:
            return qs
        
        # Allow riders to see their assigned bookings
        if hasattr(user, 'rider_profile'):
            return qs.filter(models.Q(user=user) | models.Q(rider=user.rider_profile))

        return qs.filter(user=user)

    def list(self, request, *args, **kwargs):
        qs = self.filter_queryset(self.get_queryset())
        page = self.paginate_queryset(qs)
        if page is not None:
            serializer = self.get_serializer(page, many=True)
            paginated = self.get_paginated_response(serializer.data)
            return ResponseHelper.success(data=paginated.data, message='Bookings list')

        serializer = self.get_serializer(qs, many=True)
        return ResponseHelper.success(data=serializer.data, message='Bookings list')

    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        if serializer.is_valid():
            booking = serializer.save()
            return ResponseHelper.success(data=self.get_serializer(booking).data, message='Booking created', status=status.HTTP_201_CREATED)
        return ResponseHelper.validation_error(errors=serializer.errors)


class BookingDetailView(generics.RetrieveUpdateDestroyAPIView):
    serializer_class = BookingSerializer
    permission_classes = [permissions.IsAuthenticated]

    def get_queryset(self):
        user = self.request.user
        qs = Booking.objects.all()
        if user.is_staff or user.is_superuser:
            return qs
        
        # Allow riders to see their assigned bookings
        if hasattr(user, 'rider_profile'):
            return qs.filter(models.Q(user=user) | models.Q(rider=user.rider_profile))

        return qs.filter(user=user)

    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance)
        return ResponseHelper.success(data=serializer.data, message='Booking detail')

    def update(self, request, *args, **kwargs):
        partial = kwargs.pop('partial', False)
        instance = self.get_object()
        serializer = self.get_serializer(instance, data=request.data, partial=partial)
        if serializer.is_valid():
            updated = serializer.save()
            return ResponseHelper.success(data=self.get_serializer(updated).data, message='Booking updated')
        return ResponseHelper.validation_error(errors=serializer.errors)

    def destroy(self, request, *args, **kwargs):
        instance = self.get_object()
        instance.delete()
        return ResponseHelper.success(message='Booking deleted')
